<?php
/**
 * PluginReconstructor - Plugin-Rekonstruktion beim Restore
 * 
 * Installiert Plugins basierend auf dem Inventar:
 * - WP.org Plugins: Automatisch herunterladen
 * - Premium/Custom: Aus Backup wiederherstellen
 * 
 * @package JenvaBackupMigration
 * @since 2.0.0
 */

namespace JenvaBackupMigration\Restore;

use JenvaBackupMigration\Core\Container;

if (!defined('ABSPATH')) {
    exit;
}

class PluginReconstructor {
    
    /** @var array Statistiken */
    private $stats = [];
    
    /** @var callable|null Progress-Callback */
    private $progress_callback;
    
    /**
     * Setzt den Progress-Callback
     */
    public function setProgressCallback(callable $callback): void {
        $this->progress_callback = $callback;
    }
    
    /**
     * Rekonstruiert Plugins aus dem Inventar
     * 
     * @param array $plugins Plugin-Inventar
     * @param Container $container Backup-Container
     * @param bool $keep_existing_plugins Wenn false, werden Plugins entfernt, die nicht im Backup sind
     * @return array Ergebnis
     */
    public function reconstruct(array $plugins, Container $container, bool $keep_existing_plugins = false): array {
        $this->stats = [
            'total' => count($plugins),
            'installed' => 0,
            'activated' => 0,
            'restored_from_backup' => 0,
            'skipped' => 0,
            'removed' => 0,
            'failed' => [],
        ];
        
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/plugin-install.php';
        require_once ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        
        // Schritt 1: Backup-Plugins installieren/aktivieren
        foreach ($plugins as $i => $plugin) {
            // Progress melden
            if ($this->progress_callback) {
                call_user_func($this->progress_callback, $i + 1, count($plugins), $plugin['name']);
            }
            
            // Bereits installiert?
            if ($this->isInstalled($plugin['slug'])) {
                $this->stats['skipped']++;
                
                // Aktivieren wenn nötig
                if ($plugin['active'] && !is_plugin_active($plugin['file'])) {
                    activate_plugin($plugin['file']);
                    $this->stats['activated']++;
                }
                
                continue;
            }
            
            // Installieren basierend auf Quelle
            $installed = false;
            
            if ($plugin['source'] === 'wordpress.org') {
                $installed = $this->installFromWpOrg($plugin['slug']);
            } elseif ($plugin['backed_up']) {
                $installed = $this->restoreFromBackup($plugin, $container);
            }
            
            if ($installed) {
                $this->stats['installed']++;
                
                // Aktivieren wenn nötig
                if ($plugin['active']) {
                    $result = activate_plugin($plugin['file']);
                    if (!is_wp_error($result)) {
                        $this->stats['activated']++;
                    }
                }
            } else {
                $this->stats['failed'][] = [
                    'plugin' => $plugin['name'],
                    'slug' => $plugin['slug'],
                    'source' => $plugin['source'],
                ];
            }
        }
        
        // Schritt 2: Plugins entfernen, die nicht im Backup sind (wenn nicht behalten werden sollen)
        if (!$keep_existing_plugins) {
            $this->removeUnusedPlugins($plugins);
        }
        
        return $this->stats;
    }
    
    /**
     * Entfernt Plugins, die nicht im Backup-Inventar enthalten sind
     * 
     * @param array $backup_plugins Plugin-Inventar aus Backup
     */
    private function removeUnusedPlugins(array $backup_plugins): void {
        // Liste der Backup-Plugin-Slugs erstellen
        $backup_slugs = [];
        foreach ($backup_plugins as $plugin) {
            $backup_slugs[] = $plugin['slug'];
        }
        
        // Unser eigenes Plugin darf nicht entfernt werden
        $backup_slugs[] = 'jenva-backup-migration';
        
        // Alle installierten Plugins durchgehen
        $all_plugins = get_plugins();
        
        require_once ABSPATH . 'wp-admin/includes/plugin.php';
        require_once ABSPATH . 'wp-admin/includes/file.php';
        
        foreach ($all_plugins as $plugin_file => $plugin_data) {
            $plugin_slug = dirname($plugin_file);
            
            // Single-File-Plugins haben '.' als dirname - slug ist dann der Dateiname ohne .php
            if ($plugin_slug === '.') {
                $plugin_slug = basename($plugin_file, '.php');
            }
            
            // Skip unser eigenes Plugin und Plugins die im Backup sind
            if (in_array($plugin_slug, $backup_slugs)) {
                continue;
            }
            
            // Skip WordPress Standard-Plugins (falls vorhanden)
            if (in_array($plugin_slug, ['hello', 'akismet'])) {
                continue;
            }
            
            // Plugin deaktivieren falls aktiv
            if (is_plugin_active($plugin_file)) {
                deactivate_plugins($plugin_file);
            }
            
            // Plugin-Verzeichnis löschen
            // SICHERHEIT: Path-Traversal-Schutz
            $plugin_file_dir = dirname($plugin_file);
            $real_plugin_dir = realpath(WP_PLUGIN_DIR);
            
            if ($real_plugin_dir === false) {
                continue; // WP_PLUGIN_DIR ist ungültig
            }
            
            // Single-File-Plugin: Datei direkt löschen
            if ($plugin_file_dir === '.' || $plugin_file_dir === '') {
                $plugin_path = $real_plugin_dir . '/' . basename($plugin_file);
                $real_plugin_path = realpath($plugin_path);
                
                // Path-Traversal-Schutz
                if ($real_plugin_path !== false && 
                    strpos($real_plugin_path, $real_plugin_dir) === 0 && 
                    file_exists($real_plugin_path)) {
                    @unlink($real_plugin_path);
                    $this->stats['removed']++;
                }
            } else {
                // Multi-File-Plugin: Verzeichnis rekursiv löschen
                $plugin_dir = $real_plugin_dir . '/' . $plugin_file_dir;
                $real_plugin_dir_path = realpath($plugin_dir);
                
                // Path-Traversal-Schutz: Verzeichnis muss innerhalb von WP_PLUGIN_DIR liegen
                if ($real_plugin_dir_path !== false && 
                    strpos($real_plugin_dir_path, $real_plugin_dir) === 0 && 
                    is_dir($real_plugin_dir_path)) {
                    $this->deleteDirectory($real_plugin_dir_path);
                    $this->stats['removed']++;
                }
            }
        }
    }
    
    /**
     * Löscht ein Verzeichnis rekursiv
     * 
     * @param string $dir Verzeichnis-Pfad
     * @return bool Erfolg
     */
    private function deleteDirectory(string $dir): bool {
        // SICHERHEIT: Path-Traversal-Schutz
        $real_dir = realpath($dir);
        $real_plugin_dir = realpath(WP_PLUGIN_DIR);
        
        // Prüfe ob Verzeichnis existiert und gültig ist
        if ($real_dir === false || $real_plugin_dir === false) {
            return false;
        }
        
        // Stelle sicher, dass Verzeichnis innerhalb von WP_PLUGIN_DIR liegt
        if (strpos($real_dir, $real_plugin_dir) !== 0) {
            return false; // Path Traversal erkannt
        }
        
        if (!is_dir($real_dir)) {
            return false;
        }
        
        $files = array_diff(scandir($real_dir), ['.', '..']);
        
        foreach ($files as $file) {
            // SICHERHEIT: Pfad erneut validieren nach Verkettung
            $path = $real_dir . DIRECTORY_SEPARATOR . $file;
            $real_path = realpath($path);
            
            // Path-Traversal-Schutz: Pfad muss weiterhin innerhalb von WP_PLUGIN_DIR liegen
            if ($real_path === false || strpos($real_path, $real_plugin_dir) !== 0) {
                continue; // Überspringe unsichere Pfade
            }
            
            if (is_dir($real_path)) {
                $this->deleteDirectory($real_path);
            } else {
                @unlink($real_path);
            }
        }
        
        return @rmdir($real_dir);
    }
    
    /**
     * Prüft ob Plugin installiert ist
     */
    private function isInstalled(string $slug): bool {
        $plugins = get_plugins();
        
        foreach ($plugins as $file => $data) {
            if (dirname($file) === $slug || basename($file, '.php') === $slug) {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Installiert Plugin von WordPress.org
     */
    private function installFromWpOrg(string $slug): bool {
        // Plugin-Info holen
        $api = plugins_api('plugin_information', [
            'slug' => $slug,
            'fields' => [
                'sections' => false,
                'download_link' => true,
            ],
        ]);
        
        if (is_wp_error($api)) {
            return false;
        }
        
        // Silent Upgrader
        $skin = new \WP_Ajax_Upgrader_Skin();
        $upgrader = new \Plugin_Upgrader($skin);
        
        $result = $upgrader->install($api->download_link);
        
        return $result === true;
    }
    
    /**
     * Stellt Plugin aus Backup wieder her
     */
    private function restoreFromBackup(array $plugin, Container $container): bool {
        try {
            // Plugin-Segment extrahieren
            $segments = $container->getManifest()->getSegments();
            
            foreach ($segments as $name => $info) {
                if ($info['type'] === 'plugins') {
                    $segment = $container->getSegment($name);
                    $segment->openForReading();
                    
                    // Nur dieses Plugin extrahieren
                    $files = $segment->listFiles();
                    
                    foreach ($files as $file) {
                        if (strpos($file['name'], $plugin['slug'] . '/') === 0 ||
                            strpos($file['name'], 'plugins/' . $plugin['slug'] . '/') === 0) {
                            
                            $destination = WP_PLUGIN_DIR . '/' . basename(dirname($file['name'])) . '/' . basename($file['name']);
                            $segment->extractFile($file['name'], $destination);
                        }
                    }
                    
                    $segment->close();
                    $this->stats['restored_from_backup']++;
                    return true;
                }
            }
        } catch (\Exception $e) {
            // Fehler ignorieren
        }
        
        return false;
    }
    
    /**
     * Getter für Statistiken
     */
    public function getStats(): array {
        return $this->stats;
    }
}

